#include <io.h>
#include <stdio.h>
#include <process.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/wait.h>

#define USE_REDIRECTION

static int bail (const char *msg)
{
    fprintf (stderr, "ERROR: %s: %s (%d)\n", msg, strerror(errno), errno);
    return errno;
}

int main (int argc, const char **argv)
{
    int rc;

    int fds[2];
    rc = socketpair (AF_OS2, SOCK_STREAM, 0, fds);
    if (rc == -1)
        return bail ("socketpair");

    rc = fcntl (fds[0], F_SETFD, FD_CLOEXEC);
    if (rc == -1)
        return bail ("fcntl (fds[0], FD_CLOEXEC)");
    rc = fcntl (fds[0], F_SETFL, fcntl (fds[1], F_GETFL) | O_NONBLOCK);
    if (rc == -1)
        return bail ("fcntl (O_NONBLOCK)");

    printf ("created socket pair %d, %d\n", fds[0], fds[1]);

    pid_t pid;

    if (argc == 1)
    {
        printf ("*** using sdtout redirection\n");

        int tmp_stdout_fd = dup(fileno(stdout));
        if (tmp_stdout_fd == -1)
            return bail ("dup");

        rc = dup2(fds[1], fileno(stdout));
        if (rc == -1)
            return bail ("dup2(fds[1])");

        rc = fcntl (fds[1], F_SETFD, FD_CLOEXEC);
        if (rc == -1)
            return bail ("fcntl (fds[1], FD_CLOEXEC)");

        pid = spawnlp (P_NOWAIT, "child.exe", "child.exe", NULL);

        rc = dup2(tmp_stdout_fd, fileno(stdout));
        if (rc == -1)
            return bail ("dup2(tmp_stdout_fd)");
    }
    else
    {
        printf ("*** passing fd as argument\n");

        char fdStr[10];
        sprintf (fdStr, "%d", fds[1]);
        pid = spawnlp (P_NOWAIT, "child.exe", "child.exe", fdStr, NULL);
    }

    if (pid == -1)
        return bail("spawnlp");

    close (fds[1]);

    printf ("started child.exe with PID %d\n", pid);

    fd_set readfds;
    FD_ZERO (&readfds);

    FD_SET (fds[0], &readfds);

    printf ("reading child.exe output:\n");

    bool done = false;
    char buf[256];

    do
    {
        rc = select (fds[0] + 1, &readfds, NULL, NULL, NULL);
        if (rc == -1)
            return bail ("select");

        if (!FD_ISSET (fds[0], &readfds))
            return bail ("FD_ISSET is false");

        do
        {
            rc = read(fds[0], buf, sizeof(buf));
            if (rc == -1)
                return bail ("read");
            if (rc <= 0)
                break;
            if (rc > 3 && strstr(buf, "***\n"))
                done = true;
            write(fileno(stdout), buf, rc);
        }
        while (!done);
    }
    while (!done);

    printf ("waiting for child.exe to finish...\n");

    int exitStatus;
    rc = waitpid (pid, &exitStatus, 0);
    if (pid == -1)
        return bail("waitpid");

    if (!WIFEXITED (exitStatus))
        printf ("child.exe crashed\n");
    else
        printf ("child.exe finished with code %d\n", WEXITSTATUS (exitStatus));

    return 0;
}
