//----------------------------------------------------------------------//
//                                                                      //
//                                                                      //
//                                                                      //
//                                                                      //
//----------------------------------------------------------------------//


#include "filecrypter.h"

#include <QDesktopServices>
#include <QMessageBox>
#include <QSettings>
#include <QProcess>
#include <QDir>

#include <QDebug>

//----------------------------------------------------------------------//
//                                                                      //
//----------------------------------------------------------------------//

FileCrypter::FileCrypter( const QString &tool )
{
    m_exitCode = -1;

    if( tool.isEmpty() )
    {
#ifdef  Q_OS_WIN
        QStringList pathList(
            QSettings( "HKEY_LOCAL_MACHINE\\SOFTWARE\\7-Zip",
                       QSettings::NativeFormat ).value( "Path" ).toString() );

        foreach( const QString & var, QProcess::systemEnvironment() )
        {
            int n = var.indexOf( '=' );
            if( n > 0 )
            {
                QStringRef  key( &var, 0, n );
                if( key.compare( "ProgramFiles", Qt::CaseInsensitive ) == 0 ||
                    key.compare( "ProgramW6432", Qt::CaseInsensitive ) == 0 )
                {
                    pathList.append( var.mid( n + 1 ) + "\\7-Zip" );
                }
            }
        }

        foreach( const QString & path, pathList )
        {
            if( !path.isEmpty() )
            {
                m_tool = QDir::fromNativeSeparators( path );

                if( !m_tool.endsWith( '/' ) )
                    m_tool.append( '/' );

                m_tool.append( "7z.exe" );

                if( QFile::exists( m_tool ) )
                    break;

                m_tool.clear();
            }
        }

//        qDebug() << pathList << m_tool;

#else
        m_tool = "7za";

#endif
    }
    else
    {
        m_tool = tool;
    }
}


//----------------------------------------------------------------------//
//                                                                      //
//----------------------------------------------------------------------//

void FileCrypter::setPassword( const QString &pswd )
{
    m_pswd = pswd;
}


//----------------------------------------------------------------------//
//                                                                      //
//----------------------------------------------------------------------//

bool FileCrypter::getToolVersion( QString &version )
{
    QProcess    worker;

    version.clear();

    if( m_tool.isEmpty() )
        return false;

    worker.start( m_tool, QStringList(),
                  QIODevice::ReadOnly | QIODevice::Text );

    if( worker.waitForFinished() )
    {
        while( worker.canReadLine() )
        {
            QByteArray  ba = worker.readLine();

            if( ba.contains( "Copyright" ) )
                version = ba.trimmed();
        }
    }

    return version.startsWith( "7-Zip" );
}


//----------------------------------------------------------------------//
//                                                                      //
//----------------------------------------------------------------------//

bool FileCrypter::listFiles( const QString &archiveName,
                             const QStringList &suffixes,
                             QStringList &fileList )
{
    QProcess    worker;
    QStringList args( "l" );

    m_exitCode = -1;
    fileList.clear();

    if( m_tool.isEmpty() )
        return false;

    if( !m_pswd.isEmpty() )
        args << "-p" + m_pswd;

    args << "-y" << "-slt";
    args << archiveName;

    worker.start( m_tool, args,
                  QIODevice::ReadOnly | QIODevice::Text );

    if( worker.waitForStarted() )
    {
        QByteArray  ba;
        QString     fileName;
        QString     fileAttr;
        bool        bValid = false;

        while( worker.waitForReadyRead() )
        {
            ba += worker.readAll();

            int s = 0;
            int e = ba.indexOf( '\n', s );

            while( e >= s )
            {
                QString line( ba.mid( s, e - s ) );
                int     n = line.indexOf( '=' );

//qDebug() << "7-zip dir:" << ba;

                if( n > 0 )
                {
                    QStringRef  key( &line, 0, n - 1 );

                    n += 2;

                    if( key == "Path" )                 // erster
                        fileName = line.mid( n );
                    else if( key == "Size" )
                        ;
                    else if( key == "Modified" )
                        ;
                    else if( key == "Attributes" )
                        fileAttr = line.mid( n );
                    else if( key == "Block" )           // letzter
                        bValid = true;
                }

                s = e + 1;
                e = ba.indexOf( '\n', s );

                if( bValid )
                {
                    if( !fileName.isEmpty() && fileAttr.at( 0 ) != 'D' )
                    {
                        if( suffixes.isEmpty() ||
                            suffixes.contains( QFileInfo( fileName ).suffix(),
                                               Qt::CaseInsensitive ) )
                        {
                            fileList.append( fileName );
                        }
                    }

                    fileName.clear();
                    fileAttr.clear();
                    bValid = false;
                }
            }

            if( s > 0 )
                ba.remove( 0, s );
        }

        worker.waitForFinished();

        m_exitCode = worker.exitCode();

        if( m_exitCode == 0 )
            return true;
    }

    return false;
}


//----------------------------------------------------------------------//
//                                                                      //
//----------------------------------------------------------------------//

bool FileCrypter::open( const QString &archiveName, const QString &fileName,
                        bool bExtract, bool bBinary, QProcess &worker )
{
    QStringList         args;
    QIODevice::OpenMode mode;

    m_exitCode = -1;

    if( m_tool.isEmpty() )
        return false;

    if( bExtract )
    {
        args << "e" << "-y" << "-t7z" << "-so";

        if( !m_pswd.isEmpty() )
            args << "-p" + m_pswd;

        args << QDir::toNativeSeparators( archiveName ) << fileName;

        mode = QIODevice::ReadOnly;
    }
    else
    {
        args << "a" << "-y" << "-t7z" << "-si" + fileName;

        if( !m_pswd.isEmpty() )
            args << "-p" + m_pswd;

        args << QDir::toNativeSeparators( archiveName );

        mode = QIODevice::WriteOnly;
    }

    if( !bBinary )
        mode |= QIODevice::Text;

//qDebug() << "command" << m_tool << args;

    worker.start( m_tool, args, mode );

    return worker.waitForStarted();
}


//----------------------------------------------------------------------//
//                                                                      //
//----------------------------------------------------------------------//

void FileCrypter::close( QProcess &worker )
{
    if( worker.openMode() & QIODevice::WriteOnly )
        worker.closeWriteChannel();

    worker.waitForFinished();
    m_exitCode = worker.exitCode();
}

