#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <process.h>
#include <sys/socket.h>
#include <sys/fcntl.h>
#include <errno.h>

#ifdef USE_POLL
#include <sys/poll.h>
#else
#include <sys/select.h>
#endif

//#define WORKAROUND

#ifdef WORKAROUND
static void child_exit()
{
  flushall();
}
#endif

const char child_str[] = "I'm a real child, pid 0x%04x\n";

int main_child()
{
#ifdef WORKAROUND
  atexit(child_exit);
#endif

  printf(child_str, getpid());

  return 0x11;
}

int main_parent(int argc, const char** argv)
{
  int p[2];
  int pid, status;
  ssize_t n, cnt;
#ifdef USE_POLL
  struct pollfd pfd;
#else
  fd_set readfds;
#endif
  char buf[128];

  if (socketpair(AF_UNIX, SOCK_STREAM, 0, p) < 0)
  {
    perror("socketpair");
    return 1;
  }

  pid = fork();
  if (!pid)
  {
    // child

    dup2(p[1], 1);
    close(p[0]);
    close(p[1]);

    execlp(argv[0], argv[0], "child", NULL);
    perror("exec");
    return 1;
  }

  // parent
  if (pid < 0)
  {
    perror("fork");
    return 1;
  }

  close(p[1]);

  printf("child pid 0x%x\n", pid);

  cnt = 0;

#ifdef USE_POLL
  pfd.fd = p[0];
  pfd.events = POLLIN;

  while (1)
  {
    if (poll(&pfd, 1, -1) < 0)
    {
      if (errno == EINTR)
        continue;
      perror("poll");
      break;
    }

    if (!(pfd.revents & (POLLOUT|POLLIN|POLLHUP|POLLERR|POLLNVAL)))
      continue;

    n = read(pfd.fd, buf, sizeof(buf));
    if (n <= 0)
    {
      close(pfd.fd);
      if (n < 0)
        perror("read");
      break;
    }
    else
    {
      cnt += n;
      printf("read %d bytes from child [%.*s]\n", n, n, buf);
    }
  }
#else
  while (1)
  {
    fd_set readfds;
    FD_ZERO(&readfds);
    FD_SET(p[0], &readfds);

    if (select(p[0] + 1, &readfds, NULL, NULL, NULL) < 0)
    {
      if (errno == EINTR)
        continue;
      perror("select");
      break;
    }

    if (!FD_ISSET(p[0], &readfds))
      continue;

    n = read(p[0], buf, sizeof(buf));
    if (n <= 0)
    {
      close(p[0]);
      if (n < 0)
        perror("read");
      break;
    }
    else
    {
      cnt += n;
      printf("read %d bytes from child [%.*s]\n", n, n, buf);
    }
  }
#endif

  while ((pid = waitpid(-1, &status, 0)) < 0)
  {
    if (errno == EINTR)
      continue;
    else
    {
      perror("waitpid");
      break;
    }
  }

  printf("child exit status 0x%x (pid 0x%x)\n", status, pid);

  if (cnt != (ssize_t)strlen(child_str))
  {
    printf("FAILED (read %d bytes from child instead of %d)\n", cnt, strlen(child_str));
    return 1;
  }

  printf("SUCCEEDED");
  return 0;
}

int main(int argc, const char** argv)
{
  if (argc > 1)
    return main_child();
  return main_parent(argc, argv);
}
