/****************************************************************************
*
*  var.cpp
*
*  Create:  Dec 12 2004
*  Update:  Sep 23 2005
*  Rewrite:  Sep 07 2014
*   Jan-Erik Lrka 2004 - 2014, 2017
*
****************************************************************************/
#pragma inline_depth 0;

#ifndef VAR_HPP
#define VAR_HPP

#define __STDC_WANT_LIB_EXT1__ 1

#ifndef TRUE
  #define TRUE true
#endif
#ifndef FALSE
  #define FALSE false
#endif

#ifdef __cplusplus
  using namespace std;
  #include <iostream>
#else
  #include <iostream.h>
#endif

#ifndef ULONG
  #ifndef ULONG_TYPEDEFED
    typedef unsigned long ULONG ;
    #define ULONG_TYPEDEFED
  #endif
#endif

#ifdef INCL_REXXSAA
  #define INCL_REXXSAA
  #include <rexxsaa.h>
#endif
#ifdef __REXXSAA_H_INCLUDED
#define __REXXSAA_H_INCLUDED
typedef struct RXSTRING_TYPE
{
    ULONG strlength ;
    char* strptr ;
} RXSTRING;
typedef RXSTRING *PRXSTRING ;
#endif

typedef struct ints
{
    int val[8];
    ints* ptr;
} INTS;

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

enum Calculate { Add, Sub, Mult, Div };
enum Strip { Leading, Trailing, Both };
enum Mode { Resize, Grow, Static, Reference };

/* Methods (Member functions) for class */
#define min(a,b)  (((a) < (b)) ? (a) : (b))
#define max(a,b)  (((a) > (b)) ? (a) : (b))

#ifdef __OS2__
Mode osmode = Grow;
#else
Mode osmode = Resize;
#endif

class var {
private:
    char *string; //Pointer to array that hold the character string.
    int alloc_size, alloc_mult; //Maintained value to indicate the length of the string.
    Mode resizeable; //Maintained value to indicate how memory should be initialized.
    ints *array, *e;

    int intAt( int const start_pos ) {
        return intAt( start_pos, NULL );
    };

    int intAt( int const start_pos, int value ) {
        if( array == NULL ) {
            array = new ints();
            for( int j = 0; j < 8; j++ ) array->val[ j ] = -1;
            array->ptr = NULL;
        }
        e = array;
        for( int i = 0; i < start_pos / 8; i++ ) {
            if( e->ptr == NULL ) {
                e->ptr = new ints();
                for( int j = 0; j < 8; j++ ) e->ptr->val[ j ] = -1;
                e->ptr->ptr = NULL;
            }
            e = e->ptr;
        }
        if( value == NULL )
            return e->val[ start_pos % 8 ];
        else
            return ( e->val[ start_pos % 8 ] = value );
    };

    bool init( int size_text, int reset_from = -1 ) {
        alloc_size = 0;
        size_text = size_text < 0 ? 0: size_text;
        reset_from = reset_from < 0 ? size_text : reset_from;

        if( resizeable == Reference ) return false;

        int alloc = alloc_mult;
        if( resizeable == Resize || string == NULL )
            alloc = ( size_text / 8 + 1 ) * 8;
        else if( resizeable == Grow )
            alloc = max( alloc_mult, ( size_text / 8 + 1 ) * 8 );

        if( string ) {
            if( alloc != alloc_mult ) {
                delete [] string;
                string = new char[ alloc ];
            }
        } else
            string = new char[ alloc ];
        if( string ) {
            for( int i = min( alloc - 1, reset_from ); i < alloc; i++ ) string[ i ] = '\0';
            alloc_mult = alloc;
            alloc_size = min( alloc, size_text );
            return true;
        } return false;
    };

public:
    /* Constructor alt. 1
     * var( size_t<, mode> )
     *
     * This constructor create the new object that can hold
     * a string of specified size. Additionally it's possible
     * to specify if the string/array should be static in size
     * or dynamically resizeable by the class.
     * Parameters:
     *                size_text - number of characters (mandatory)
     *                mode - how to allocate memory (mandatory)
     */
    var( int size_text, Mode mode ) : resizeable( mode ), string( NULL ), array( NULL ), e( NULL ) {
        init( size_text, 0 );
    };

    /** Copy constructor
     * var( class<<<,start_pos = 0> ,size_text = -1> , mode = Grow > )
     *
     * This copy constructor copy contence so it doesn't point to
     * same address space. Additionally it's possible to specify
     * if the target string/array should be dynamically resizeable.
     * Parameters:
     *                v - variable of class var with string (mandatory)
     *                start_pos - start copying from (optional)
     *                size_text - copy n characters. (optional)
     *                mode - Grow by default on OS/2-eCS, follow length by default on Windows (optional)
     */
    var( var const &v, int start_pos = 0, int size_text = -1, Mode mode = osmode ) : resizeable( mode ), string( NULL ), array( NULL ), e( NULL ) {
        cpy( v, start_pos, size_text );
    };

    /** Constructor alt. 2 (Default constructor too)
     * var( <<<text>, size = -1>, static size = true> )
     *
     * This constructor create the new object that can hold
     * a string of specified size. Additionally it's possible
     * to specify if the string/array should be dynamically
     * resizeable. If no parameters used it act as default
     * constructor and just prepare for further use.
     * Parameters:
     *                text - string to set at creation (optional)
     *                size_text - number of characters (optional)
     *                mode - Grow by default on OS/2-eCS, follow length by default on Windows (optional)
     */
    var( const char *text = NULL, int size_text = -1, Mode mode = osmode ) : resizeable( mode ), string( NULL ), array( NULL ), e( NULL ) {
        if ( size_text < 0 )
            size_text = length( text );
        if( init( size_text ) )
            for( int i = 0; i < size_text && text[ i ]; i++ ) string[ i ] = text[ i ];
    };

    /** Constructor alt. 3
     * var( text<, size_text = 1> )
     *
     * This constructor create the new object that can hold
     * a string of specified size. Additionally it's possible
     * to specify if the string/array should be dynamically
     * resizeable.
     * Parameters:
     *                text - string to set at creation (mandatory)
     *                size_text - number of characters (optional)
     *                mode - Grow by default on OS/2-eCS, follow length by default on Windows (optional)
     */
    var( char text, int size_text = 1, Mode mode = osmode ) : resizeable( mode ), string( NULL ), array( NULL ), e( NULL ) {
        if( init( text != '\0' ? size_text : 0 ) )
            for( int i = 0; i < size_text; i++ ) string[ i ] = text;
    };

    /** Constructor alt. 4
     * var( integer )
     *
     * This constructor create the new object that can hold
     * a value originating from an integer.
     * Parameters:
     *                i - integer to convert to var (mandatory)
     */
    var( int i ) : resizeable( osmode ), string( NULL ), array( NULL ), e( NULL ) {
        if( init( 17 ) )
            itoa( i, string, 10 );
        reSize( -1, resizeable );
    };

    /** Constructor alt. 5
     * var( double )
     *
     * This constructor create the new object that can hold
     * a value originating from a double.
     * Parameters:
     *                d - double to set at creation (mandatory)
     *                mode - Grow by default on OS/2-eCS, follow length by default on Windows (optional)
     */
    var( double d, Mode mode = osmode ) : resizeable( mode ), string( NULL ), array( NULL ), e( NULL ) {
        if( init( 35 ) )
            gcvt( d, 10, string );
        reSize( -1, resizeable );
    };

    /** Constructor alt. 6
     * var( unsigned long )
     *
     * This constructor create the new object that can hold
     * a value originating from an unsigned long.
     * Parameters:
     *                ul - unsigned long to set at creation (mandatory)
     *                mode - Grow by default on OS/2-eCS, follow length by default on Windows (optional)
     */
    var( unsigned long ul, Mode mode = osmode ) : resizeable( mode ), string( NULL ), array( NULL ), e( NULL ) {
        if( init( 17 ) )
            ltoa( ul, string, 10 );
        reSize( -1, resizeable );
    };

    /** Constructor alt. 7
     * var( <text, size_text> )
     *
     * This constructor just reuse an already allocated string
     * Parameters:
     *                text - string to set at creation (required)
     */
    var( char *text, int size_text ) : resizeable( Reference ), array( NULL ), e( NULL ) {
        if ( size_text < 0 )
            alloc_size = length( text );
        else
            alloc_size = size_text;
        alloc_mult = alloc_size;
        string = text;
    };

    /** Destructor
     * ~var()
     *
     * Destructor used to destroy allocated object and clean up memory.
     * Parameters:
     *                none
     */
    ~var() {
        delete [] string;
        while( e != NULL ) {
            e = array->ptr;
            delete array;
            array = e;
        }
    };

    //The following member functions are declared:
    bool isNumeric() const { if( string[ 0 ] == '0' ) return TRUE; return atof( string ) != 0; };
    bool isInteger() const { int i = atoi( string ); return ( ( i != 0 ) || ( string[ 0 ] == '0' && i == 0 ) ); };
    bool isAlpha() const { if( alloc_size == 0 ) return false; double d = atof( string ); var temp( 35, Static ); gcvt( d, 10, temp ); return temp.length( false ) < alloc_size; }
    bool isEmpty() const { return alloc_size == 0; };

    var& reSize( int size_text = -1, Mode mode = osmode, bool truncate = true ) {

        size_text = ( size_text < 0 ? length( false ) : size_text );
        size_text = ( size_text < alloc_size ? ( truncate ? size_text : alloc_size ): size_text );
        int alloc = alloc_mult;

        if( resizeable == Resize || string == NULL )
            alloc = ( size_text / 8 + 1 ) * 8;
        else if( resizeable == Grow )
            alloc = max( alloc_mult, ( size_text / 8 + 1 ) * 8 );

        int i = 0;
        if ( alloc_mult != alloc && resizeable != Static && resizeable != Reference ) { //Resize to
            alloc_mult = alloc;
            char* temp = new char[ alloc_mult ]; // Create a temporary string
            if( temp != NULL ) {
                for( ; i < min( alloc_size, size_text ); i++ ) temp[ i ] = string[ i ]; // Copy parts of string to temp storage
                delete [] string; // Delete string
                string = temp;
            }
        } else i = size_text;
        alloc_size = size_text;
        //for( ; i < alloc_mult; i++ )
        string[ i ] = '\0';
        resizeable = mode;
        return ( *this );
    };

    var& flush() {
        for( int i = alloc_size; i < alloc_mult; i++ ) string[ i ] = '\0';
        return *this;
    }

    operator char const *() { return string; };
    operator char *() { return string; };
    operator char **() { return &string; };
    operator unsigned char *() { return (unsigned char *) string; };
    operator unsigned char **() { return (unsigned char **) string; };
    operator char() const { return string[0]; };
    operator int() { return atoi( string ); }; // Automatically pretend/appear as int if needed.
    operator long() { return atol( string ); }; // Automatically pretend/appear as long if needed.
    operator double() { return atof( string ); }; // Automatically pretend/appear as double if needed.
    operator unsigned long() { return strtoul( string, NULL, 10 ); }; // Automatically appear as ulong if needed.

    var &operator =( var const &v ) {
        init( v.length() );
        for( int i = 0; i < v.length(); i++ ) string[ i ] = v.string[ i ]; // Copy string
        return ( *this );
    };
    var &operator =( char const *s ) {
        int strlen = length( s );
        init( strlen );
        for( int i = 0; i < strlen; i++ ) string[ i ] = s[ i ]; // Copy string
        return ( *this );
    };
    var &operator =( char const c ) {
        if( init( 1 ) )
            string[ 0 ] = c;
        return ( *this );
    };
    var &operator =( int const i ) {
        (*this) = fromInt( i );
        return ( *this );
    };
    var &operator =( double const d ) {
        (*this) = fromDouble( d );
        return ( *this );
    };
    var &operator =( unsigned long const ul ) {
        (*this) = fromULONG( ul );
        return ( *this );
    };

    var &operator +=( var const &v ) {
        calculate( v, Add );
        return *this;
    };
    var &operator +=( char const *s ) {
        calculate( s, Add );
        return *this;
    };
    var &operator +=( char const c ) {
        reSize( length() + 1, resizeable );
        string[ length() + 1 ] = c;
        return ( *this );
    };
    var &operator +=( int const i ) {
        calculate( fromInt( i ), Add );
        return ( *this );
    };
    var &operator +=( double const d ) {
        calculate( fromDouble( d ), Add );
        return ( *this );
    };
    var &operator +=( unsigned long const ul ) {
        calculate( fromULONG( ul ), Add );
        return ( *this );
    };

    var operator + ( char const *s ) const {
        var retval( *this );
        return retval += s;
    };
    var operator + ( char const c ) const {
        var retval( *this );
        return retval += c;
    };
    var operator + ( var const &v ) const {
        var retval( *this );
        return retval += v;
    };
    var operator + ( int const i ) const {
        var retval( *this );
        return retval += i;
    };
    var operator + ( double const d ) const {
        var retval( *this );
        return retval += d;
    };
    var operator + ( unsigned long const ul ) const {
        var retval( *this );
        return retval += ul;
    };

    var &operator -=( var const &v ) {
        calculate( v, Sub );
        return *this;
    };
    var &operator -=( char const *s ) {
        calculate( s, Sub );
        return *this;
    };
    var &operator -=( char const c ) {
        remove( c );
        return ( *this );
    };
    var &operator -=( int const i ) {
        calculate( fromInt( i ), Sub );
        return ( *this );
    };
    var &operator -=( double const d ) {
        calculate( fromDouble( d ), Sub );
        return ( *this );
    };
    var &operator -=( unsigned long const ul ) {
        calculate( fromULONG( ul ), Sub );
        return ( *this );
    };

    var operator - ( char const *s ) const {
        var retval( *this );
        return retval -= s;
    };
    var operator - ( char const c ) const {
        var retval( *this );
        return retval -= c;
    };
    var operator - ( var const &v ) const {
        var retval( *this );
        return retval -= v;
    };
    var operator - ( int const i ) const {
        var retval( *this );
        return retval -= i;
    };
    var operator - ( double const d ) const {
        var retval( *this );
        return retval -= d;
    };
    var operator - ( unsigned long const ul ) const {
        var retval( *this );
        return retval -= ul;
    };

    var &operator *=( var const &v ) {
        calculate( v, Mult );
        return *this;
    };
    var &operator *=( char const *s ) {
        calculate( s, Mult );
        return *this;
    };

    var &operator *=( int const i ) {
        calculate( fromInt( i ), Mult );
        return ( *this );
    };
    var &operator *=( double const d ) {
        calculate( fromDouble( d ), Mult );
        return ( *this );
    };
    var &operator *=( unsigned long const ul ) {
        calculate( fromULONG( ul ), Mult );
        return ( *this );
    };

    var operator * ( char const *s ) const {
        var retval( *this );
        return retval *= s;
    };
    var operator * ( char const c ) const {
        var retval( *this );
        return retval *= c;
    };
    var operator * ( var const &v ) const {
        var retval( *this );
        return retval *= v;
    };
    var operator * ( int const i ) const {
        var retval( *this );
        return retval *= i;
    };
    var operator * ( double const d ) const {
        var retval( *this );
        return retval *= d;
    };
    var operator * ( unsigned long const ul ) const {
        var retval( *this );
        return retval *= ul;
    };

    var operator /=( var const &v ) {
        var retval( calculate( v, Div ) );
        return retval;
    };
    var operator /=( char const *s ) {
        var retval( calculate( s, Div ) );
        return retval;
    };
    var operator /=( char const c ) {
        int i = bruteForceSearch( c );
        var retval;
        if( i < 0 ) return retval;
        retval = split( i, 1 );
        return retval;
    };
    var operator /=( int const i ) {
        var retval( calculate( fromInt( i ), Div ) );
        return retval;
    };
    var operator /=( double const d ) {
        var retval( calculate( fromDouble( d ), Div ) );
        return retval;
    };
    var operator /=( unsigned long const ul ) {
        var retval( calculate( fromULONG( ul ), Div ) );
        return retval;
    };

    var operator / ( char const *s ) {
        int i = boyerMooreSearch( s );
        if( i > -1 )
            return split( i, length( s ) );
        return "";
    };
    var operator / ( char const c ) {
        int i = bruteForceSearch( c );
        if( i > -1 )
            return split( i, 1 );
        return "";
    };
    var operator / ( var const &v ) {
        int i = boyerMooreSearch( v );
        if( i > -1 )
            return split( i, v.length() );
        return "";
    };
    var operator / ( int const num ) {
        var v( fromInt( num ) );
        int i = boyerMooreSearch( v );
        if( i > -1 )
            return split( i, v.length() );
        return "";
    };
    var operator / ( double const d ) {
        var v( fromDouble( d ) );
        int i = boyerMooreSearch( v );
        if( i > -1 )
            return split( i, v.length() );
        return "";
    };
    var operator / ( unsigned long const ul ) {
        var v( fromULONG( ul ) );
        int i = boyerMooreSearch( v );
        if( i > -1 )
            return split( i, v.length() );
        return "";
    };

    var &operator %=( var const &v ) {
        split( v );
        return *this;
    };
    var &operator %=( char const *s ) {
        int i = reverseBoyerMooreSearch( s );
        if( i > -1 )
            split( i, length( s ) );
        return *this;
    };
    var &operator %=( char const c ) {
        int i = reverseBruteForceSearch( c );
        if( i > -1 )
            split( i, 1 );
        return *this;
    };
    var &operator %=( int const i ) {
        var retval( fromInt( i ) );
        int num = reverseBoyerMooreSearch( retval );
        if( num > -1 )
            split( num, retval.length() );
        return *this;
    };
    var &operator %=( double const d ) {
        var retval( fromDouble( d ) );
        int i = reverseBoyerMooreSearch( retval );
        if( i > -1 )
            split( i, retval.length() );
        return *this;
    };
    var &operator %=( unsigned long const ul ) {
        var retval( fromULONG( ul ) );
        int i = reverseBoyerMooreSearch( retval );
        if( i > -1 )
            split( i, retval.length() );
        return *this;
    };

    var operator % ( char const *s ) {
        return split( s );
    };
    var operator % ( char const c ) {
        return split( c );
    };
    var operator % ( var const &v ) {
        return split( v );
    };
    var operator % ( int const i ) {
        return split( fromInt( i ) );
    };
    var operator % ( double const d ) {
        return split( fromDouble( d ) );
    };
    var operator % ( unsigned long const ul ) {
       return split( fromULONG( ul ) );
    };
    var& operator ++ () {
        return calculate( "1", Add );
    };
    var& operator ++ ( int ) {
        return calculate( "1", Add );
    };

    var& operator -- () {
        return calculate( "1", Sub );
    };
    var& operator -- ( int ){
        return calculate( "1", Sub );
    };

/*    bool operator == ( char const *s ) const {
        return ( match( s ) == 0 );
    }

    bool operator == ( char const c ) const {
        return ( match( c ) == 0 );
    }

    bool operator == ( var const &v ) const {
        return ( match( v ) == 0 );
    }

    bool operator == ( int const i ) const {
        return ( match( fromInt( i ) ) == 0 );
    }

    bool operator == ( unsigned long const ul ) const {
        return ( match( fromULONG( ul ) ) == 0 );
    }

    bool operator == ( double const d ) const {
        return ( match( fromDouble( d ) ) == 0 );
    }

    bool operator != ( char const *s ) const {
        return ( match( s ) != 0 );
    }

    bool operator != ( char const c ) const {
        return ( match( c ) != 0 );
    }

    bool operator != ( var const &v ) const {
        return ( match( v ) != 0 );
    }

    bool operator != ( int const i ) const {
        return ( match( fromInt( i ) ) != 0 );
    }

    bool operator != ( unsigned long const ul ) const {
        return ( match( fromULONG( ul ) ) != 0 );
    }

    bool operator != ( double const d ) const {
        return ( match( fromDouble( d ) ) != 0 );
    }*/


    var operator ()( int const start_pos = 0, int const size_text = -1 ) const {
        return cpy( start_pos, size_text );
    };
    char &operator ()( int const start_pos = 0 ) {
        return string[ start_pos > 0 ? max( 0, start_pos ) % alloc_mult : 0 ];
    };
    char const &operator []( int const start_pos ) const {
        return string[ alloc_size > 0 ? max( 0, start_pos ) % alloc_mult : 0 ];
    };
    char &operator []( int const start_pos ) {
        return string[ alloc_size > 0 ? max( 0, start_pos ) % alloc_mult : 0 ];
    };

    int operator !() const { return ( alloc_size > 0 ); };
    int length() const { return alloc_size; };
    int length( bool allocated = true ) {
        if( allocated ) return alloc_size;
        alloc_size = 0;
        const char *p = string;
        while( *p && p != '\0' ) { p++; alloc_size++; }
        return alloc_size; // Return number of characters

    };
    int length( char const *text ) {
        int size = 0;
        if( text ) {
            const char *p = text;
            while( *p && p != '\0' ) { p++; size++; }
        }
        return size; // Return number of characters
    };
    char const &get_at( int start_pos ) const {
        return string[ alloc_size > 0 ? max( 0, start_pos ) % alloc_mult: 0 ];
    };
    void put_at( int start_pos, char c ) {
        if( -1 < start_pos && start_pos < alloc_mult )
            string[ max( 0, start_pos ) % alloc_mult ] = c;
    };
    int index( var const &v, int start_pos = -1 ) const { return boyerMooreSearch( v, start_pos ); };
    var upper() const {
        var retval( *this );
        for( int i = 0; i < retval.length(); i++ ) retval.string[i] = toupper( retval.string[i] );
        return retval;
    };
    var lower() const {
        var retval( *this );
        for( int i = 0; i < retval.length(); i++ ) retval.string[i] = tolower( retval.string[i] );
        return retval;
    };
    int valid() const { return ( string != NULL ); };
    int alloc_mult_size() const { return alloc_mult; };
    int alloc_mult_size( int text_size ) { reSize( text_size, resizeable ); return alloc_mult; };

    //The following friend functions are declared:

    friend int   operator ==( var const &v1, var const &v2 ) { return v1.match( v2 ) == v1.length(); };
    friend int   operator ==( var const &v,   char const *s ) { return v.match( s ) == v.length(); };
    friend int   operator ==(   char const *s, var const &v ) { return v.match( s ) == v.length(); };
    friend int   operator ==( var const &v,   char c        ) { return v.match( c ) == v.length(); };
    friend int   operator ==(   char c      , var const &v ) { return v.match( c ) == v.length(); };
    friend int   operator !=( var const &v1, var const &v2 ) { return v1.match( v2 ) < v1.length(); };
    friend int   operator !=( var const &v,   char const *s ) { return v.match( s ) < v.length(); };
    friend int   operator !=(   char const *s, var const &v ) { return v.match( s ) > v.length(); };
    friend int   operator !=( var const &v,   char c        ) { return v.match( c ) < v.length(); };
    friend int   operator !=(   char c      , var const &v ) { return v.match( c ) > v.length(); };
    friend int   operator <( var const &v1, var const &v2 ) { return v1.match( v2 ) < 0; };
    friend int   operator <( var const &v,   char const *s ) { return v.match( s ) < 0; };
    friend int   operator <(   char const *s, var const &v ) { return v.match( s ) > 0; };
    friend int   operator <( var const &v,   char c        ) { return v.match( c ) < 0; };
    friend int   operator <(   char c      , var const &v ) { return v.match( c ) > 0; };
    friend int   operator <=( var const &v1, var const &v2 ) { return v1.match( v2 ) <= v1.length(); };
    friend int   operator <=( var const &v,   char const *s ) { return v.match( s ) <= v.length(); };
    friend int   operator <=(   char const *s, var const &v ) { return v.match( s ) > v.length(); };
    friend int   operator <=( var const &v,   char c        ) { return v.match( c ) <= v.length(); };
    friend int   operator <=(   char c      , var const &v ) { return v.match( c ) > v.length(); };
    friend int   operator >( var const &v1, var const &v2 ) { return v1.match( v2 ) > 0; };
    friend int   operator >( var const &v,   char const *s ) { return v.match( s ) > 0; };
    friend int   operator >(   char const *s, var const &v ) { return v.match( s ) < 0; };
    friend int   operator >( var const &v,   char c        ) { return v.match( c ) > 0; };
    friend int   operator >(   char c      , var const &v ) { return v.match( c ) < 0; };
    friend int   operator >=( var const &v1, var const &v2 ) { return v1.match( v2 ) >= v1.length(); };
    friend int   operator >=( var const &v,   char const *s ) { return v.match( s ) >= v.length(); };
    friend int   operator >=(   char const *s, var const &v ) { return v.match( s ) < v.length(); };
    friend int   operator >=( var const &v,   char c        ) { return v.match( c ) >= v.length(); };
    friend int   operator >=(   char c      , var const &v ) { return v.match( c ) < v.length(); };

/*    friend var operator +(       var &v1, var const &v2 ) {
        var retval( v1 );
        return retval += v2;
    };
    friend var operator +(       var &v,   char const *s ) {
        var retval( v );
        return retval += s;
    };
    friend var operator +(   char const *s, var const &v ) {
        var retval( s );
        return retval += v;
    };*/
    friend var operator +( int const , var const & );
    //friend var operator +( unsigned long const , const var const & );
    //friend var operator +( double const , const var const & );
    friend int valid( var const &v ) { return ( v.string != NULL ); };

    //The following I/O Stream inserter and extractor functions are declared:

    friend istream &operator >>( istream &in, var &v ) { in >> v.string; return in; };
    friend ostream &operator <<( ostream &out, var const &v ) { out << v.string; return out; };

    var& calculate( var const &v, Calculate type, bool magic = false ) {
        if( !isAlpha() || magic ) {
            double this_num = atof( string ), v_num = atof( v.string );
            var this_pre( 35, Grow ), v_pre( 35, Grow ), this_post( 35, Grow );
            gcvt( this_num, 10, this_pre );
            gcvt( v_num, 10, v_pre );
            int v_len = v_pre.length( false );
            switch( type ) {
            case Sub:
                this_num -= v_num;
                break;
            case Div:
                if( v_num > 0 ) this_num /= v_num;
                break;
            case Mult:
                this_num *= v_num;
                break;
            default:
                this_num += v_num;
            }
            if( this_num > 0 && v_num > 0 ) {
                gcvt( this_num, 10, this_post );
                int this_pre_len = this_pre.length( false ), this_post_len = this_post.length( false );
                this_post.reSize( this_post_len + length() - this_pre_len, resizeable );
                for( int i = this_post_len, j = 0; j < min( this_post.alloc_size - this_pre_len, this_post.alloc_mult ); i++, j++ ) this_post[ i ] = string[ this_pre_len + j ]; // Copy parts of string to temp storage
                reSize( this_post.length(), resizeable );
                for( int i = 0; i < min( this_post.length(), alloc_size ); i++ ) string[ i ] = this_post[ i ]; // Copy to string
                if( type == Add )
                    merge( v( v_len ) );
                else if( type == Sub )
                    remove( v( v_len ) );
                else if( type == Div )
                    split( v( v_len ) );
                else if( type == Mult )
                    repeat( v( v_len ) );

/*                int this_pre_len = length( this_pre ), this_post_length = this_post.reSize( -1, resizeable ).length();
                int size_text = this_post_length + alloc_size - this_pre_len + v.length() - v_len;

                if( resizeable == Grow )
                    alloc_mult = max( alloc_mult, ( size_text / 8 + 1 ) * 8 );
                else if( resizeable == Resize )
                    alloc_mult = ( size_text / 8 + 1 ) * 8;

                if ( size_text > alloc_mult )
                    size_text = alloc_mult - 1;

                char* temp = new char[ alloc_mult ]; // Create a temporary string
                int i = 0, j;
                if( temp != NULL ) {
                    for( j = 0; j < this_post_length && i < min( size_text, alloc_mult ); i++, j++ ) temp[ i ] = this_post.string[ j ]; // Copy parts of string to temp storage
                    for( j = 0; j < alloc_size - this_pre_len && i < min( size_text, alloc_mult ); i++, j++ ) temp[ i ] = string[ this_pre_len + j ]; // Copy parts of string to temp storage
                    for( j = 0; j < v.length() - v_len && i < min( size_text, alloc_mult ); i++, j++ ) temp[ i ] = v.string[ v_len + j ]; // Copy parts of string to temp storage
                    delete [] string; // Delete string
                    string = temp;
                    alloc_size = size_text;
                }
                for( int i = size_text; i < alloc_mult; i++ ) string[ i ] = '\0';*/
            }
        } else if( type == Add )
            merge( v );
        else if( type == Sub )
            remove( v );
        else if( type == Div )
            split( v );
        else if( type == Mult )
            repeat( v );
        return *this;
    };

    var fromInt( const int i ) const {
        char retval[ 17 ];
        itoa( i, retval, 10 );
        return retval;
    }
    var fromULONG( const ULONG ul ) const {
        char retval[ 17 ];
        ltoa( ul, retval, 10 );
        return retval;
    }
    var fromDouble( const double d ) const {
        char retval[ 35 ];
        gcvt( d, 10, retval );
        return retval;
    }
#ifdef INCL_REXXSAA
    var& set( RXSTRING &rxString ) {
        if( rxString.strptr )
            delete rxString.strptr;
        rxString.strptr = new char[ alloc_size + 1 ];
        if( rxString.strptr ) {
            for( int i = 0; i <= alloc_size; i++ ) rxString.strptr[ i ] = string[ i ];
            rxString.strlength = alloc_size;
        } else {
            rxString.strptr = 0;
            rxString.strlength = 0;
        }
        return ( *this );
    }
    var& set( RXSTRING *rxString ) {
        if( rxString->strptr )
            delete rxString->strptr;
        rxString->strptr = new char[ alloc_size + 1 ];
        if( rxString->strptr ) {
            for( int i = 0; i <= alloc_size; i++ ) rxString->strptr[ i ] = string[ i ];
            rxString->strlength = alloc_size;
        } else {
            rxString->strptr = 0;
            rxString->strlength = 0;
        }
        return ( *this );
    }

    var& flush( RXSTRING &rxString ) {
        rxString.strptr = 0;
        rxString.strlength = 0;
        return ( *this );
    }
    var& get( PRXSTRING rxString ) {
        if( rxString->strptr && rxString->strlength )
            cpy( rxString->strptr, 0, rxString->strlength );
        return ( *this );
    }
#endif

    var &reverse() {
        char temp;
        char *pEnd, *pStart;

        pEnd = string;
        pStart = string;
        while( *pEnd ) pEnd++;
        while( pEnd > pStart ) {
            pEnd--;
            temp = *pEnd;
            *pEnd = *pStart;
            *pStart = temp;
            pStart++;
        }
        return ( *this );
    }

    var &cat( const var &v1, const var &v2 = '\0', const var &v3 = '\0', const var &v4 = '\0', const var &v5 = '\0', const var &v6 = '\0', const var &v7 = '\0', const var &v8 = '\0', const var &v9 = '\0', const var &v10 = '\0', const var &v11 = '\0', const var &v12 = '\0' ) {
        int i1 = v1.length(), i2 = v2.length(), i3 = v3.length();
        int i4 = v4.length(), i5 = v5.length(), i6 = v6.length();
        int i7 = v7.length(), i8 = v8.length(), i9 = v9.length();
        int i10 = v10.length(), i11 = v11.length(), i12 = v12.length();
        int i = length( false ), j;
        reSize( i + i1 + i2 + i3 + i4 + i5 + i6 + i7 + i8 + i9 + i10 + i11 + i12, resizeable );
        for( j = 0; j < i1 && i < alloc_mult; string[ i++ ] = v1.string[ j++ ] );
        for( j = 0; j < i2 && i < alloc_mult; string[ i++ ] = v2.string[ j++ ] );
        for( j = 0; j < i3 && i < alloc_mult; string[ i++ ] = v3.string[ j++ ] );
        for( j = 0; j < i4 && i < alloc_mult; string[ i++ ] = v4.string[ j++ ] );
        for( j = 0; j < i5 && i < alloc_mult; string[ i++ ] = v5.string[ j++ ] );
        for( j = 0; j < i6 && i < alloc_mult; string[ i++ ] = v6.string[ j++ ] );
        for( j = 0; j < i7 && i < alloc_mult; string[ i++ ] = v7.string[ j++ ] );
        for( j = 0; j < i8 && i < alloc_mult; string[ i++ ] = v8.string[ j++ ] );
        for( j = 0; j < i9 && i < alloc_mult; string[ i++ ] = v9.string[ j++ ] );
        for( j = 0; j < i10 && i < alloc_mult; string[ i++ ] = v10.string[ j++ ] );
        for( j = 0; j < i11 && i < alloc_mult; string[ i++ ] = v11.string[ j++ ] );
        for( j = 0; j < i12 && i < alloc_mult; string[ i++ ] = v12.string[ j++ ] );
        return *this;
    }

    var split( int start_pos = 0, int skip = 0 ) { //Split and return string( s )
        start_pos = min( max( 0, start_pos ), alloc_size );
        skip = max( 0, skip );
        var retval( alloc_size - start_pos - skip, resizeable );
        for( int i = 0; i < retval.length(); i++ ) { retval.string[ i ] = string[ start_pos + skip + i ]; string[ start_pos + skip + i ] = '\0'; }
        reSize( start_pos, resizeable );
        return retval;
    }

    var split( const var& v ) { //Split and return string( s )
        int i = boyerMooreSearch( v );
        i = ( i > -1 ) ? i : v.length() - i;
        return split( i, v.length() );
    }

    var split( const char c, int skip, bool exact ) {
        int i = bruteForceSearch( c, -1, exact );
        if ( i > -1 )
            return split( i, abs( skip ) );
        return "";
    }

    var remove( int start_pos, int len = 0 ) { // Return string
        int i = 0;
        int txtlen = min( alloc_size, abs( len ) + abs( start_pos ) );
        var retval( txtlen, resizeable );
        if( retval.string ) {
            for( ; i < start_pos; i++ ) retval.string[ i ] = string[ i ];
            for( ; i < txtlen; i++ ) retval.string[ i ] = string[ i + len ];
        }
        return retval;
    }

    var &remove( const char c ) { // Return string
        int i = bruteForceSearch( c );
        int j = i, k = 0;
        while ( i > -1 ) {
            for( ; j + k < i && j + k <= alloc_size; j++ ) string[ j ] = string[ j + k ];
            k++;
            i = bruteForceSearch( c, i + 1 );
        }
        for( ; j + k <= alloc_size; j++ ) string[ j ] = string[ j + k ];
        reSize( alloc_size - k + 1, resizeable );
        return *this;
    }

    var &remove( const var& v ) { //Remove text
        return replace( v, "" );
    }

    var &replace( const var& v, const var& v2 ) { // text
        int M = v.length();
        int newM = v2.length();
        int i = 0, j = 0, k = 0, l = 0,count = 0;
        int new_alloc_mult = alloc_mult, new_alloc_size = alloc_size;
        char* temp;

        int skip[ 256 ];
        for ( i = 0; i < 256; ++i ) skip[ i ] = M;
        for ( i = 0; i < M; ++i ) skip[ v.string[ i ] ] = M - 1 - i;

        i = M - 1;
        for( ; -2 < j && i < alloc_size; ) {
            for ( j = M - 1; 0 <= j; --i, --j )
                while ( string[ i ] != v.string[ j ] ) {
                    i += max( M - j, skip[ string[ i ] ] );
                    if( alloc_size <= i )
                    {
                        j = -2;
                        break;
                    }
                    j = M - 1;
                }
            if( -2 < j )
                i = intAt( count++, i + 1 ) + 1 + M;
        }
        if( newM - M != 0 && resizeable != Static && resizeable != Reference ) {
            new_alloc_size = alloc_size + count * ( newM - M );
            new_alloc_mult = ( new_alloc_size / 8 + 1 ) * 8;
            temp = new char[ new_alloc_mult ]; // Create a temporary string
        } else
            temp = string;
        if( temp != NULL && 0 < count ) {
            i = 0, j = 0, l = 0;
            for( ; i < count; i++ ) {
                for( ; l < intAt( i ); )
                    temp[ j++ ] = string[ l++ ];
                for( k = 0; k < newM; )
                    temp[ j++ ] = v2.string[ k++ ];
                l += M;
            }
            while( e != NULL ) {
                e = array->ptr;
                delete array;
                array = e;
            }
            if( newM - M != 0 ) {
                for( ; l < alloc_size; ) temp[ j++ ] = string[ l++ ];
                for( ; j < new_alloc_mult; ) temp[ j++ ] = '\0';
                if( resizeable != Static && resizeable != Reference ) {
                    delete [] string; // Delete string
                    string = temp;
                }
                alloc_size = new_alloc_size;
                alloc_mult = new_alloc_mult;
            }
        }
        return ( *this );
    }

    var repeat( int count ) {
        var retval( *this );
        for( int i = 0; i < abs( count ); i++ )
            retval += (*this);
        return retval;
    }

    var repeat( const var& v ) {
        var retval( *this );
        retval += v;
        return retval;
    }

    int boyerMooreSearch( const var& needle, int start_pos = -1 ) const {
        int i, j;
        int skip[256];
        int M = needle.length();

        for ( i = 0; i < 256; ++i ) skip[ i ] = M;
        for ( i = 0; i < M; ++i ) skip[ needle.string[ i ] ] = M - 1 - i;

        start_pos = start_pos < 0 ? M : min( max( start_pos, M ), alloc_size ) - 1;

        for ( i = start_pos, j = M - 1; j >= 0; --i, --j )
            while ( string[ i ] != needle.string[ j ] ) {
                i += max( M - j, skip[ string[ i ] ] );
                if ( i >= length() ) return -1;
                j = M - 1;
            }
        return i + 1;
    }

    int reverseBoyerMooreSearch( const var& needle, int start_pos = -1 ) const {
        int i, j;
        int skip[256];
        int M = needle.length();

        for ( i = 0; i < 256; ++i ) skip[ i ] = M;
        for ( i = 0; i < M; ++i ) skip[ needle.string[ i ] ] = M /*+ 1*/ - i;

        start_pos = start_pos < 0 ? alloc_size - M : min( start_pos, alloc_size - M );

        for ( i = start_pos, j = 0; j < M; ++i, ++j ) {
            while ( string[ i ] != needle.string[ j ] ) {
                i -= max( j, skip[ string[ i ] ] );
                if ( i < 0 ) return -1;
                j = 0;
            }
        }
        return i - M;
    }

    var& strip( Strip type = Both, const char needle = ' ', bool exact = true ) {
        int fromStart = 0, fromEnd = alloc_size;
        if( type != Trailing )
            fromStart = max( 0, match( needle, exact ) );
        if( type != Leading )
            fromEnd = max( 0, match( needle, exact, true ) );
        fromEnd -= fromStart;
        if( fromStart > 0 )
            for( int i = 0; i < fromEnd; i++ ) string[ i ] = string[ i + fromStart ];
        if( fromEnd != alloc_size )
            reSize( fromEnd, resizeable );
        return (*this);
    }

    int reverseBruteForceSearch( const char needle = ' ', int fromEnd = -1, bool caseSensetive = false ) const {
        int i = length() - ( fromEnd < 0 ? 0 : min( length(), fromEnd ) );
        if ( caseSensetive )
            for( i--; i > -1; i-- ) { if( string[ i ] == needle ) return i; }
        else
            for( i--; i > -1; i-- ) { if( toupper( string[ i ] ) == toupper( needle ) ) return i; }
        return -1;
    }

    int bruteForceSearch( const char needle = ' ', int start = -1, bool caseSensetive = false ) const {
        int i = ( start < 0 ) ? 0 : min( start, length() ), stop = length();
        if ( caseSensetive )
            for( ; i < stop; i++ ) { if( string[ i ] == needle ) return i; }
        else
            for( ; i < stop; i++ ) { if( toupper( string[ i ] ) == toupper( needle ) ) return i; }
        return -1;
    }

    int match( var const &v, bool caseSensetive = false ) const {
        double this_double = atof( string ), v_double = atof( v.string );
        if( this_double != 0 && v_double != 0 )
            return this_double < v_double;
        int i = 0, stop = min( alloc_size, v.length() );
        if ( caseSensetive )
            while( string[ i ] == v.string[ i ] && i < stop ) i++;
        else
            while( toupper( string[ i ] ) == toupper( v.string[ i ] ) && i < stop  ) i++;
        return ( ( ( int )( string[ i ] ) > ( int )( v.string[ i ] ) ) ? i : -i );
    }

    int match( char const c, bool caseSensetive = false, bool reverse = false ) const {
        int pos = reverse ? alloc_size: 0;
        if( string ) {
            const char *p = string;
            if( caseSensetive ) {
                if( reverse )
                    while( *p && pos > -1 && ( string[ --pos ] != c || string[ pos ] == '\0' ) )
                        cout << string[ pos ];
                else
                    while( *p && string[ pos ] != c && string[ pos ] != '\0' ) { p++; pos++; }
            } else {
                char cup = toupper( c );
                if( reverse )
                    while( *p && pos > -1 && ( toupper( string[ --pos ] ) != cup || string[ pos ] == '\0' ) )
                        cout << string[ pos ];
                else
                    while( *p && toupper( string[ pos ] ) != cup && string[ pos ] != '\0' ) { p++; pos++; }
            }
        }
        return pos + reverse; // Return last pos of matching characters
    }

    char* fromLast( var const c ) {
        return string + match( c, false, true );
    }

    var &toLast( var const c ) {
        for( int i = match( c, false, true ); i < alloc_size; i++ ) string[ i ] = '\0';
        return *this;
    }

    var &before( var const c, bool last = false ) {
        for( int i = match( c, false, last ) - 1; i < alloc_size; i++ ) string[ i ] = '\0';
        return *this;
    }

    char* after( var const c, bool last = false ) {
        return string + match( c, false, last ) + 1;
    }

    bool contains( const char* needle ) {
        const char* haystack = string;
        do {
            const char* h = haystack;
            const char* n = needle;
            while( toupper( (unsigned char) *h ) == toupper( (unsigned char) *n ) && *n ) {
                h++;
                n++;
            }
            if ( *n == 0 )
                return true;
        } while( *haystack++ );
        return false;
    }

    bool wordBeginsWith( const var& needle, const var& delimiter = " ", bool caseSensetive = false ) const {
        int count = words( delimiter );
        for( int i = 0; i < count; i++ ) {
            if( subWord( 0, 1, delimiter ).beginsWith( needle, 0, caseSensetive ) ) return true;
        }
        return false;
    }

    bool beginsWith( const var& needle, int start = 0, bool caseSensetive = false ) const {
        int strlen = length(), findlen = needle.length();
        int i = ( start < 0 ) ? 0 : min( start, strlen );
        for( ; i < strlen && i < findlen ; i++ ) {
            if( caseSensetive ) {
                if( string[ i ] != needle.string[ i ] ) goto done;
            } else {
                if( toupper( string[ i ] ) != toupper( needle[ i ] ) ) goto done;
            }
        }
    done:
        return ( i >= ( start + findlen ) );
    }

    var subWord( int start_word = 0, int nr_words = -1, const var& delimiters = " " ) const { //Return part (word) of string
        int word, a, b, i = 0, j = 0, k = 0, strlen = length(), dellen = delimiters.length(), word_count = ( nr_words < 0 ) ? words( delimiters ) : abs( nr_words + start_word );

        for( word = 0; word < start_word && i < strlen; word++ ) { // Word count
            for( i = k; i < strlen && j < dellen; i++ ) // Spaces
                for( j = 0; string[ i ] != delimiters.string[ j ] && j < dellen; j++ ); // Delimiters
            for( k = i + 1; k < strlen && j == dellen; k++ ) // Text to form Word
                for( j = 0; string[ k ] != delimiters.string[ j ] && j < dellen; j++ ); // Delimiters
        }
        for( a = b = k; word < word_count && a < strlen; word++ ) { // Word count
            for( a = b; a < strlen && j < dellen; a++ ) // Spaces
                for( j = 0; string[ a ] != delimiters.string[ j ] && j < dellen; j++ ); // Delimiters
            for( b = a + 1; b < strlen && j == dellen; b++ ) // Text to form Word
                for( j = 0; string[ b ] != delimiters.string[ j ] && j < dellen; j++ ); // Delimiters
        }
        return cpy( max( a - 1 , 0 ), max( ( b == 0 ) ? strlen : b - ( b < strlen ), 0 ) );
    }

    int wordPos( const var &v, const var& delimiters = " ", bool caseSensetive = false ) const { //Return position of part (word) in string
        int word = 0, i = 0, j = 0, dellen = delimiters.length(), strlen = length(), vlen = v.length();
        for( ; i < strlen; word++ ) { // Word count
            for( j = 0; string[ i ] != delimiters.string[ 0 ] && i < strlen; ) {
                    if( caseSensetive ) {
                        for( int k = 0; string[ i++ ] == v.string[ k++ ] && i < strlen && k < vlen; )
                            if( k + 1 == vlen )
                                return word;
                    } else {
                        for( int k = 0; toupper( string[ i++ ] ) == toupper( v.string[ k++ ] ) && i < strlen && k < vlen; )
                            if( k + 1 == vlen )
                                return word;
                    }
            }
            for( j = 0; string[ i++ ] == delimiters.string[ j++ ] && i < strlen && j < dellen; ); // Delimiters
            if( j < dellen ) word--;
        }
        return word;
    }


    int wordPosIn( var &v, const var& delimiters = " ", bool caseSensetive = false ) const { //Return position of part (word) in string
        return v.wordPos( *this, delimiters, caseSensetive );
    }

    int words( const char delimiter = ' ' ) const {
        int word = 0, i = 0, strlen = length();
        for( ; i < strlen; i++ ) {
            if( delimiter == string[ i ] ) word++;
        }
        if ( strlen > 0 )
            if( delimiter != string[ i - 1 ] ) word++;
        return word;
    }

    int words( const var& delimiters ) const {
        int word = 0, i = 0, j = 0, dellen = delimiters.length(), strlen = length();
        for( ; i < strlen; word++ ) { // Word count
            for( ; i < strlen & j < dellen; i++ ) // Spaces
                for( j = 0; string[ i ] != delimiters.string[ j ] && j < dellen; j++ ); // Delimiters
            for( ; i < strlen & j == dellen; i++ ) // Text to form Word
                for( j = 0; string[ i ] != delimiters.string[ j ] && j < dellen; j++ ); // Delimiters
            if( j < dellen ) word--;
        }
        return word;
    }

    var cpy( int start_pos = 0, int size_text = -1 ) const {
        start_pos = min( length(), max( 0, start_pos ) );
        size_text = size_text < 0 ? alloc_size - min( alloc_size, start_pos ) : size_text;
        var retval( size_text, resizeable );
        if( string != NULL )
            for( int i = 0; i < size_text; i++ ) retval.string[ i ] = string[ start_pos + i ];
        return retval;
    };

    var &cpy( var const &v, int start_pos = 0, int size_text = -1 ) {
        int i, v_len = v.length();
        start_pos = min( v_len, max( 0, start_pos ) );
        if( size_text < 0 || v_len < size_text )
            size_text = v_len - start_pos;
        if( init( size_text, min( v_len, size_text ) ) )
            for( i = 0; i < min( v_len, size_text ); i++ ) string[ i ] = v[ start_pos + i ];
        return *this;
    };

    var &merge( var const &v, int start_pos = 0, int size_text = -1 ) { //Join strings
        start_pos = ( start_pos < 0 ) ? 0 : start_pos;
        int strlen = ( ( size_text < 0 ) ? v.length() : size_text ) - start_pos, alloc = alloc_size;
        reSize( alloc + strlen, resizeable );
        if( string ) {
            for( int i = 0; i < strlen; i++ )
                string[ alloc + i ] = v.string[ start_pos + i ];
            alloc_size = alloc + strlen;
        }
        return *this;
    };

    var &left( int text_size = -1 ) {
        var temp;
        return temp.cpy( *this, 0, text_size < 0 ? alloc_size : text_size );
    }

    var &right( int text_size = 0 ) {
        var temp;
        return temp.cpy( *this, text_size, alloc_size );
    }
};
#endif
