#include <stdlib.h>

#include <qapplication.h>
#include <qwidget.h>
#include <qpainter.h>
#include <qdatetime.h>

/*
 * This test calculates the roundtrip speed of copy offscreen buffer to window
 * operations. Besides the time necessary for copying pixels, this also includes
 * the time necessary to fill 2D rectangles in the offscreen buffer, deliver
 * paint and timer messages and so on.
 *
 * To also get the speed of only the copy operatoins themselves, add
 *
 *      #define QT_LOG_BLITSPEED
 *
 * to the beginning of qapplication_pm.cpp and qwindowsurface_raster.cpp.
 *
 * Hint: in order to get the most realistic numbers, let the test run until the
 * average speed printed in the window title stabilizes (i.e. stops to grow).
 **/

////////////////////////////////////////////////////////////////////////////////

class MyWidget : public QWidget
{
public:

    MyWidget() : mUpdateTID(0), mStatTID(0), mNumPixels(0)
    {
    };

    ~MyWidget()
    {
        unsigned long long elapsed = mElapsedTimer.elapsed();
        qWarning("Stats:");
        qWarning("  total pixels blitted           : %llu", mNumPixels);
        qWarning("  total time, ms                 : %llu", elapsed);
        qWarning("  average roundtrip speed, px/ms : %llu", mNumPixels / elapsed);
    }

    void startWork()
    {
        mElapsedTimer.start();
        mUpdateTID = startTimer(0);
        mStatTID = startTimer(500);
    }

    void paintEvent(QPaintEvent *aE)
    {
        QPainter p(this);

        QRect r = aE->rect();

        QColor c = QColor(rand() / (RAND_MAX / 255),
                          rand() / (RAND_MAX / 255),
                          rand() / (RAND_MAX / 255));
        p.fillRect(r, c);

        mNumPixels += r.width() * r.height();
    }

    void timerEvent(QTimerEvent *aE)
    {
        if (aE->timerId() == mUpdateTID) {
            int w = width();
            int h = height();
            int x = w < 2 ? 0 : rand() / (RAND_MAX / (w - 1));
            int y = h < 2 ? 0 : rand() / (RAND_MAX / (w - 1));
            w -= x;
            h -= y;
            w = w < 2 ? 1 : rand() / (RAND_MAX / w);
            h = h < 2 ? 1 : rand() / (RAND_MAX / h);

            update(x, y, w, h);
        } else if (aE->timerId() == mStatTID) {
            unsigned long long elapsed = mElapsedTimer.elapsed();
            QString title = "Blit Stats: %1 pixels/ms";
            title = title.arg(mNumPixels / elapsed);
            setCaption(title);
        }
    }

private:

    int mUpdateTID;
    int mStatTID;
    unsigned long long mNumPixels;
    QTime mElapsedTimer;
};

int main(int argc, char **argv)
{
    QApplication app(argc, argv);
    app.connect(&app, SIGNAL(lastWindowClosed()), SLOT(quit()));

    srand(QDateTime::currentDateTime().toTime_t());

    MyWidget widget;
    widget.resize(300, 300);
    widget.show();

    QRect sr = QApplication::desktop()->availableGeometry();
    QRect gr = widget.frameGeometry();
    widget.move(sr.width() - gr.width(), sr.height() - gr.height());

    widget.startWork();

    return app.exec();
}

