//----------------------------------------------------------------------//
//									//
//	Direct screen access for Qt's raster surface on OS/2		//
//									//
//----------------------------------------------------------------------//

#define		INCL_PM
#define		INCL_DOS
#include	<os2.h>
#include	"dive.h"

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<malloc.h>


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

class OS2DiveBlitter
{
    HWND		hwnd;

    HDIVE		hDive;
    RECTL		rcScreen;
    ULONG		cbScanLine;
    ULONG		ctBitsPerPixel;
    FOURCC		fccScreen;

    PBYTE		pFrameBuffer;

    static ULONG	arMapTable[3][256];
    static ULONG	refCount;
    static HMODULE	hmodDive;


public:
    OS2DiveBlitter();
    ~OS2DiveBlitter();

    BOOL	open(HWND hWindow);
    VOID	close(BOOL fShutdown);

    BOOL	blit(HPS hps, PBYTE pBits,
		     PBITMAPINFOHEADER2 pBmih,
		     PRECTL rcSrcUpdate = NULL,
		     int nDstOffsX = 0, int nDstOffsY = 0);
};


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

#ifndef		TESTBED

extern "C" {
  BOOL APIENTRY WinSetVisibleRegionNotify(HWND hwnd, BOOL fEnable);
  ULONG APIENTRY WinQueryVisibleRegion(HWND hwnd, HRGN hrgn);
}

#ifndef	mmioFOURCC
  #define mmioFOURCC( ch0, ch1, ch2, ch3 )                         	\
		  ( (ULONG)(BYTE)(ch0) | ( (ULONG)(BYTE)(ch1) << 8 ) |	\
		  ( (ULONG)(BYTE)(ch2) << 16 ) | ( (ULONG)(BYTE)(ch3) << 24 ) )

  #define FOURCC_BGR4  mmioFOURCC( 'B', 'G', 'R', '4' )
  #define FOURCC_RGB4  mmioFOURCC( 'R', 'G', 'B', '4' )
  #define FOURCC_BGR3  mmioFOURCC( 'B', 'G', 'R', '3' )
  #define FOURCC_RGB3  mmioFOURCC( 'R', 'G', 'B', '3' )
  #define FOURCC_R565  mmioFOURCC( 'R', '5', '6', '5' )
  #define FOURCC_R555  mmioFOURCC( 'R', '5', '5', '5' )
  #define FOURCC_R664  mmioFOURCC( 'R', '6', '6', '4' )
#endif


#ifdef	__EMX__
  extern inline unsigned bswap32_p(unsigned u)
  {
    __asm__ __volatile__ ("bswap %0\n"
			  : "=r" (u)
			  : "0" (u));
    return u;
  }

#else
  #define bswap32_p(a)	((((ULONG)(a)) >> 24) | (((ULONG)(a)) << 24) | \
			 (((ULONG)(a) << 8) & 0x00ff0000) | (((ULONG)(a) >> 8) & 0x0000ff00))
#endif



//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

static ULONG (APIENTRY *pfnDiveQueryCaps)(PDIVE_CAPS pDiveCaps, ULONG ulPlaneBufNum);
static ULONG (APIENTRY *pfnDiveOpen)(HDIVE *phDiveInst, BOOL fNonScreenInstance, PVOID ppFrameBuffer);
static ULONG (APIENTRY *pfnDiveClose)(HDIVE hDiveInst);
static ULONG (APIENTRY *pfnDiveAcquireFrameBuffer)(HDIVE hDiveInst, PRECTL prectlDst);
static ULONG (APIENTRY *pfnDiveSwitchBank)(HDIVE hDiveInst, ULONG ulBankNumber);
static ULONG (APIENTRY *pfnDiveDeacquireFrameBuffer)(HDIVE hDiveInst);


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

ULONG	OS2DiveBlitter::refCount = 0;
HMODULE OS2DiveBlitter::hmodDive = NULLHANDLE;
ULONG	OS2DiveBlitter::arMapTable[3][256];


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

OS2DiveBlitter::OS2DiveBlitter()
{
    ULONG	u;
    DIVE_CAPS	Caps;

    static struct { PCSZ pchName; PFN *ppEntry; } arLoadTable[] =
    {
	{ (PCSZ)"DiveQueryCaps",	    (PFN *)&pfnDiveQueryCaps },
	{ (PCSZ)"DiveOpen",		    (PFN *)&pfnDiveOpen },
	{ (PCSZ)"DiveClose",		    (PFN *)&pfnDiveClose },
	{ (PCSZ)"DiveAcquireFrameBuffer",   (PFN *)&pfnDiveAcquireFrameBuffer },
	{ (PCSZ)"DiveSwitchBank",	    (PFN *)&pfnDiveSwitchBank },
	{ (PCSZ)"DiveDeacquireFrameBuffer", (PFN *)&pfnDiveDeacquireFrameBuffer },
    };

    if( !hmodDive && DosLoadModule(NULL, 0, (PCSZ)"DIVE", &hmodDive) == 0 )
    {
	for( u = 0; u < sizeof(arLoadTable) / sizeof(arLoadTable[0]); u++ )
	{
	    if( DosQueryProcAddr(hmodDive, 0,
			arLoadTable[u].pchName, arLoadTable[u].ppEntry) != 0 )
	    {
		hmodDive = NULLHANDLE;
		break;
	    }
	}
    }

    if( hmodDive )	refCount++;

    ctBitsPerPixel = 0;
    hDive = NULLHANDLE;
    memset(arMapTable, 0, sizeof(arMapTable));

    memset(&Caps, 0, sizeof(Caps));
    Caps.ulStructLen = sizeof(Caps);

    if( refCount &&
	pfnDiveQueryCaps(&Caps, DIVE_BUFFER_SCREEN) == DIVE_ERR_INSUFFICIENT_LENGTH )
    {
	printf("DiveCaps.fScreenDirect: %d\n", (int)Caps.fScreenDirect);
	printf("DiveCaps.fBankSwitched: %d\n", (int)Caps.fBankSwitched);
	printf("DiveCaps.ulWidth: %d\n", (int)Caps.ulHorizontalResolution);
	printf("DiveCaps.ulHeight: %d\n", (int)Caps.ulVerticalResolution);
	printf("DiveCaps.ulDepth: %d\n", (int)Caps.ulDepth);
	printf("DiveCaps.ulScanLineBytes: %d\n", (int)Caps.ulScanLineBytes);
	printf("DiveCaps.fccColorEncoding: %.4s\n", (char *)&Caps.fccColorEncoding);

	if( Caps.fScreenDirect && !Caps.fBankSwitched )
	{
	    cbScanLine       = Caps.ulScanLineBytes;
	    ctBitsPerPixel   = Caps.ulDepth;
	    rcScreen.xLeft   = 0;
	    rcScreen.yBottom = 0;
	    rcScreen.xRight  = Caps.ulHorizontalResolution;
	    rcScreen.yTop    = Caps.ulVerticalResolution;
	    fccScreen        = Caps.fccColorEncoding;

	    switch( fccScreen )
	    {
	      case FOURCC_R565:
		for( u = 0; u < 256; u++ )
		{
		   arMapTable[0][u] = (u >> 3) << 0;
		   arMapTable[1][u] = (u >> 2) << 5;
		   arMapTable[2][u] = (u >> 3) << 11;
		}
		break;

	      case FOURCC_R555:
		for( u = 0; u < 256; u++ )
		{
		   arMapTable[0][u] = (u >> 3) << 0;
		   arMapTable[1][u] = (u >> 3) << 5;
		   arMapTable[2][u] = (u >> 3) << 10;
		}
		break;

	      case FOURCC_R664:
		for( u = 0; u < 256; u++ )
		{
		   arMapTable[0][u] = (u >> 2) << 0;
		   arMapTable[1][u] = (u >> 2) << 6;
		   arMapTable[2][u] = (u >> 4) << 12;
		}
		break;

	      case FOURCC_BGR4:
	      case FOURCC_RGB4:
	      case FOURCC_BGR3:
	      case FOURCC_RGB3:
		break;

	      default:
		ctBitsPerPixel = 0;
		break;
	    }
	}
    }
}


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

OS2DiveBlitter::~OS2DiveBlitter()
{
    close(TRUE);

    if( --refCount == 0 )
    {
	DosFreeModule(hmodDive);
	hmodDive = NULLHANDLE;
    }
}


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

BOOL OS2DiveBlitter::open(HWND hWindow)
{
    if( ctBitsPerPixel <= 8 )
    {
       return FALSE;
    }

    if( !hDive && pfnDiveOpen(&hDive, FALSE, &pFrameBuffer) != 0 )
    {
       return FALSE;
    }

    hwnd = hWindow;

    return TRUE;
}


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

VOID OS2DiveBlitter::close(BOOL fShutdown)
{
    if( fShutdown && hDive )
    {
	pfnDiveClose(hDive);
	hDive = NULLHANDLE;
    }

    hwnd = NULLHANDLE;
}


//----------------------------------------------------------------------//
//									//
//----------------------------------------------------------------------//

BOOL OS2DiveBlitter::blit(HPS hps, PBYTE pBits,
			  PBITMAPINFOHEADER2 pBmih,
			  PRECTL rcSrcUpdate, int nDstOffsX, int nDstOffsY)
{
  HRGN		hrgn;
  ULONG		u, rc;
  RGNRECT	RgnCtl;
  PUSHORT	pusTemp;
  PRECTL	arRgnRects;
  PUCHAR	pDst, pSrc;
  RECTL		rcWindow, rcClip;
  LONG		cyWndHeight, ctRows, ctCols, i;

//  if( pBmih->cbFix < FIELDOFFSET(BITMAPINFOHEADER2, ulCompression) )
  if( pBmih->cbFix < (ULONG) (&((PBITMAPINFOHEADER2)NULL)->ulCompression) )
  {
     return FALSE;
  }

  if( pBmih->cPlanes != 1 || pBmih->cBitCount != 32 )
  {
     return FALSE;
  }


  if( hwnd && (hrgn = GpiCreateRegion(hps, 0, NULL)) != NULLHANDLE )
  {
      if( WinQueryVisibleRegion(hwnd, hrgn) != RGN_ERROR )
      {
	GpiQueryClipBox(hps, &rcClip);
	rcClip.xRight++;
	rcClip.yTop++;

	RgnCtl.ircStart    = 1;
	RgnCtl.crc         = 0;
	RgnCtl.crcReturned = 0;
	RgnCtl.ulDirection = RECTDIR_LFRT_BOTTOP;

	if( GpiQueryRegionRects(hps, hrgn, &rcClip, &RgnCtl, NULL) )
	{
//printf("VisibleRgn: %u rects\n", RgnCtl.crcReturned);

	   if( RgnCtl.crcReturned )
	   {
	     arRgnRects = (PRECTL)alloca(RgnCtl.crcReturned * sizeof(RECTL));
	     RgnCtl.crc = RgnCtl.crcReturned;
	     RgnCtl.crcReturned = 0;

	     GpiQueryRegionRects(hps, hrgn, &rcClip, &RgnCtl, arRgnRects);

//printf("VisibleRgn2: %u rects\n", RgnCtl.crcReturned);

	     WinQueryWindowRect(hwnd, &rcWindow);
	     WinMapWindowPoints(hwnd, HWND_DESKTOP, (PPOINTL)&rcWindow, 2);
	     cyWndHeight = rcWindow.yTop - rcWindow.yBottom;

	     rc = pfnDiveAcquireFrameBuffer(hDive, &rcScreen);
	     if( rc == 0 )
	     {
//	       printf("--- %d,%d  %d,%d\n", rcScreen.xLeft, rcScreen.yBottom, rcScreen.xRight, rcScreen.yTop);

		for( u = 0; u < RgnCtl.crcReturned; u++ )
		{
		  // I don't know, why...
		  arRgnRects[u].xRight--;
		  arRgnRects[u].yBottom++;

		  arRgnRects[u].xLeft  -= nDstOffsX;
		  arRgnRects[u].xRight -= nDstOffsX;
		  arRgnRects[u].yTop    = cyWndHeight - arRgnRects[u].yTop - nDstOffsY;
		  arRgnRects[u].yBottom = cyWndHeight - arRgnRects[u].yBottom - nDstOffsY;
/*
printf("  %u: %d,%d  %d,%d\n", u,
       arRgnRects[u].xLeft, arRgnRects[u].yTop,
       arRgnRects[u].xRight, arRgnRects[u].yBottom);
*/
		  if( arRgnRects[u].xLeft < rcSrcUpdate->xLeft )
			arRgnRects[u].xLeft = rcSrcUpdate->xLeft;

		  if( arRgnRects[u].xRight > rcSrcUpdate->xRight )
			arRgnRects[u].xRight = rcSrcUpdate->xRight;

		  if( arRgnRects[u].yTop < rcSrcUpdate->yTop )
			arRgnRects[u].yTop = rcSrcUpdate->yTop;

		  if( arRgnRects[u].yBottom > rcSrcUpdate->yBottom )
			arRgnRects[u].yBottom = rcSrcUpdate->yBottom;

/*
printf("  %u: %d,%d  %d,%d\n", u,
       arRgnRects[u].xLeft, arRgnRects[u].yTop,
       arRgnRects[u].xRight, arRgnRects[u].yBottom);
*/

		  ctRows = arRgnRects[u].yBottom - arRgnRects[u].yTop + 1;
		  ctCols = arRgnRects[u].xRight - arRgnRects[u].xLeft + 1;

//  printf("Rect%u: Rows: %d, Cols: %d\n", u, ctRows, ctCols);

		  if( ctCols > 0 && ctRows > 0 && pBits )
		  {
		    pSrc = (PUCHAR)&pBits[arRgnRects[u].xLeft * 4 +
				  arRgnRects[u].yTop * pBmih->cx * 4];

		    pDst = (PUCHAR)&pFrameBuffer[
			     ((rcWindow.xLeft + arRgnRects[u].xLeft + nDstOffsX) * ctBitsPerPixel) / 8 +
			     (rcScreen.yTop - rcWindow.yTop + arRgnRects[u].yTop + nDstOffsY) * cbScanLine];

		    switch( fccScreen )
		    {
		    case FOURCC_BGR4:
			do
			{
			    for( i = 0; i < ctCols; i++ )
			    {
				*(PULONG)pDst = *(PULONG)pSrc;
				pSrc += 4;	pDst += 4;
			    }

			    pSrc += (pBmih->cx - ctCols) * 4;
			    pDst += cbScanLine - (ctCols * 4);

			} while( --ctRows );
			break;

		    case FOURCC_RGB4:
			do
			{
			    for( i = 0; i < ctCols; i++ )
			    {
				*(PULONG)pDst = bswap32_p(*(PULONG)pSrc) >> 8;
				pSrc += 4;	pDst += 4;
			    }

			    pSrc += (pBmih->cx - ctCols) * 4;
			    pDst += cbScanLine - (ctCols * 4);

			} while( --ctRows );
			break;

		    case FOURCC_BGR3:
			do
			{
			    for( i = ctCols; i >= 4; i -= 4 )
			    {
				*(PULONG)&pDst[0] = (*(PULONG)pSrc & 0x00ffffff) |
						    (*(PUCHAR)&pSrc[4] << 24);
				*(PULONG)&pDst[4] = (*(PUSHORT)&pSrc[5]) |
						    (*(PUSHORT)&pSrc[8] << 16);
				*(PULONG)&pDst[8] = (*(PUCHAR)&pSrc[10]) |
						    (*(PULONG)&pSrc[12] << 8);

				pDst += 12;	pSrc += 16;
			    }

			    while( i-- )
			    {
				pDst[0] = pSrc[0];
				pDst[1] = pSrc[1];
				pDst[2] = pSrc[2];
				pDst += 3;	pSrc += 4;
			    }

			    pSrc += (pBmih->cx - ctCols) * 4;
			    pDst += cbScanLine - (ctCols * 3);

			} while( --ctRows );
			break;

		    case FOURCC_RGB3:
			do
			{
			    for( i = ctCols; i >= 4; i -= 4 )
			    {
				*(PULONG)&pDst[0] = (*(PUCHAR)&pSrc[6] << 24) |
						    (bswap32_p(*(PULONG)pSrc) >> 8);
				*(PULONG)&pDst[4] = bswap32_p(*(PUSHORT)&pSrc[9]) |
						    (bswap32_p(*(PUSHORT)&pSrc[4]) >> 16);
				*(PULONG)&pDst[8] = (*(PUCHAR)&pSrc[8]) |
						    (bswap32_p(*(PULONG)&pSrc[12]) & 0xffffff00);

				pDst += 12;	pSrc += 16;
			    }

			    while( i-- )
			    {
				pDst[2] = pSrc[0];
				pDst[1] = pSrc[1];
				pDst[0] = pSrc[2];
				pDst += 3;	pSrc += 4;
			    }

			    pSrc += (pBmih->cx - ctCols) * 4;
			    pDst += cbScanLine - (ctCols * 3);

			} while( --ctRows );
			break;

		    default:	// R565, R555, R664
			do
			{
			    pusTemp = (PUSHORT)pDst;
			    for( i = 0; i < ctCols; i++ )
			    {
			       *pusTemp++ = (USHORT)(arMapTable[0][pSrc[0]] |
						     arMapTable[1][pSrc[1]] |
						     arMapTable[2][pSrc[2]]);
			       pSrc += 4;
			    }

			    pSrc += (pBmih->cx - ctCols) * 4;
			    pDst += cbScanLine;

			} while( --ctRows );
			break;
		    }
		  }
		}

		pfnDiveDeacquireFrameBuffer(hDive);
	     }
	   }

	}
      }

      GpiDestroyRegion(hps, hrgn);
    }

    return TRUE;
}


#endif



