#include <qapplication.h>
#include <qwidget.h>
#include <qpushbutton.h>
#include <qbitmap.h>
#include <qpainter.h>
#include <qframe.h>
#include <qhbox.h>
#include <qobjectlist.h>

// -----------------------------------------------------------------------------

class MyPushButton : public QPushButton
{
public:

MyPushButton (const QString &text, const QColor &color,
              QWidget *parent, const char *name)
    : QPushButton (text, parent, name)
    , msk (false)
{
    setPaletteBackgroundColor (color);
    setAutoMask (true);
}

void setMasked (bool masked) { msk = masked; updateMask(); }
bool masked () const { return msk; }

void updateMask()
{
    if (msk)
    {
        QBitmap bmp (width(), height(), true);
        {
            QPainter p (&bmp);
            p.setPen (color1);
            p.setBrush (color1);
            p.drawEllipse (0, 0, width(), height());
        }
        setMask (bmp);
    }
    else
        clearMask();
}

private:

    bool msk;
};

// -----------------------------------------------------------------------------

class MySimpleWidget : public QWidget
{
public:

MySimpleWidget (const QColor &color, QWidget *parent, const char *name)
    : QWidget (parent, name, WStaticContents | WRepaintNoErase | WResizeNoErase)
    , clr (color), msk (false), pos (3, 3)
{
    setAutoMask (true);
}

void setMasked (bool masked) { msk = masked; updateMask(); }
bool masked () const { return msk; }

bool isLogging();

void scroll (int dx, int dy)
{
    pos.rx() += dx;
    pos.ry() += dy;
    QWidget::scroll (dx, dy);
}

void scroll (int dx, int dy, const QRect & r)
{
    pos.rx() += dx;
    pos.ry() += dy;
    QWidget::scroll (dx, dy, r);
}

void paintEvent (QPaintEvent *pe)
{
    if (isLogging())
    {
        QRect r = pe->region().boundingRect();
        qDebug ("=== paintEvent(%s/%s): rgn_bounds=%d,%d;%d,%d",
                name(), className(), r.x(), r.y(), r.width(), r.height());
    }
    
    QPainter p (this);
    p.fillRect (rect(), clr);
    p.drawText (pos.x(), pos.y() + p.fontMetrics().ascent(), name());
}

void updateMask()
{
    if (msk)
    {
        QBitmap bmp (width(), height(), true);
        {
            QPainter p (&bmp);
            p.setPen (color1);
            p.setBrush (color1);
            p.drawEllipse (0, 0, width(), height());
        }
        setMask (bmp);
    }
    else
        clearMask();
}

void enterEvent (QEvent *)
{
    if (isLogging())
    {
        qDebug ("=== enterEvent(%s/%s)",
                name(), className());
    }
}

void leaveEvent (QEvent *)
{
    if (isLogging())
    {
        qDebug ("=== leaveEvent(%s/%s)",
                name(), className());
    }
}

private:

    QColor clr;
    bool msk;
    QPoint pos;
};

// -----------------------------------------------------------------------------

class MyWidget : public QWidget
{
    Q_OBJECT

public:

MyWidget() //: QWidget( 0, 0, WStaticContents )
{
    bgPixmap = QPixmap ("marble.png");
    setErasePixmap (bgPixmap);
    
    logging = true;
    
    resize (300, 300);
    move (800, 300);
    
    // simple widgets
    
    m0 = new MySimpleWidget (yellow, this, "My0");
    m0->resize (40, 40);
    m0->move (90, 90);

    m1 = new MySimpleWidget (red, this, "My1");
    m1->resize (40, 40);
    m1->move (120, 120);

    m11 = new MySimpleWidget (cyan, m1, "My11");
    m11->resize (10, 10);
    m11->move (10, 20);

    m2 = new MySimpleWidget (green, this, "My2");
    m2->resize (40, 40);
    m2->move (120, 120);

    m3 = new MySimpleWidget (green, this, "My3");
    m3->resize (10, 10);
    m3->move (100, 100);

    // buttons widgets

    b0 = new MyPushButton ("Push0", yellow, this, "Push0");
    b0->resize (40, 40);
    b0->move (90, 90);

    b1 = new MyPushButton ("Push1", red, this, "Push1");
    b1->resize (40, 40);
    b1->move (120, 120);

    b11 = new MyPushButton ("Push11", cyan, b1, "Push11");
    b11->resize (10, 10);
    b11->move (10, 10);

    b2 = new MyPushButton ("Push2", green, this, "Push2");
    b2->resize (40, 40);
    b2->move (120, 120);

    b3 = new MyPushButton ("Push3", green, this, "Push3");
    b3->resize (10, 10);
    b3->move (100, 100);
    
    w1 = 0;
    
    setFocusPolicy (StrongFocus);
    
    switchWidgets();
}

bool isLogging() { return logging; }

void switchWidgets()
{
    if (w1 != m1)
    {
        m1->setGeometry (b1->geometry());
        w1 = m1;
        m0->show(); m1->show(); m2->show(); m3->show();
        b0->hide(); b1->hide(); b2->hide(); b3->hide();
    }
    else
    {
        b1->setGeometry (m1->geometry());
        w1 = b1;
        b0->show(); b1->show(); b2->show(); b3->show();
        m0->hide(); m1->hide(); m2->hide(); m3->hide();
    }

    setFocus();
}

void switchMaskStatus()
{
    if (m1->masked())
    {
        m0->setMasked (false); m1->setMasked (false); m11->setMasked (false);
        m2->setMasked (false); m3->setMasked (false);
        b0->setMasked (false); b1->setMasked (false); b11->setMasked (false);
        b2->setMasked (false); b3->setMasked (false);
    }
    else
    {
        m0->setMasked (true); m1->setMasked (true); m11->setMasked (true);
        m2->setMasked (true); m3->setMasked (true);
        b0->setMasked (true); b1->setMasked (true); b11->setMasked (true);
        b2->setMasked (true); b3->setMasked (true);
    }
}

void paintEvent (QPaintEvent *pe)
{
    if (isLogging())
    {
        QRect r = pe->region().boundingRect();
        qDebug ("=== paintEvent(%s/%s): rgn_bounds=%d,%d;%d,%d",
                name(), className(), r.x(), r.y(), r.width(), r.height());
    }
    QWidget::paintEvent (pe);
}

void resizeEvent (QResizeEvent *re)
{
    int rw = re->size().width();
    int rh = re->size().height();
    int w = rw / 3;
    int h = rh / 3;
    w1->setGeometry ((rw - w) / 2, (rh - h) / 2, w, h );
}

void keyPressEvent (QKeyEvent *ke)
{
    if (ke->type() == QEvent::KeyRelease ||
        ke->state() != ke->stateAfter())
        return;

    if (ke->key() == Key_Space)
    {
        qDebug ("--------------------------------------");
        return;
    }
    
    int dx = 0;
    int dy = 0;
    
    switch (ke->key())
    {
        case Key_Left: dx = -1; break;
        case Key_Right: dx = 1; break;
        case Key_Up: dy = -1; break;
        case Key_Down: dy = 1; break;
    }
    
    if (dx || dy)
    {
        if (ke->state() & QEvent::ControlButton)
        {
            resize (width() + 10 * dx, height() + 10 * dy);
        }
        else
        if ((ke->state() & QEvent::AltButton) && w1 == m1)
        {
            if (ke->state() & QEvent::ShiftButton)
                m1->scroll (1 * dx, 1 * dy,
                            QRect (0, 0, m1->width(), m1->height()));
            else
                m1->scroll (1 * dx, 1 * dy);
        }
        else
        if (ke->state() & QEvent::ShiftButton)
        {
            w1->resize (w1->width() + 5 * dx, w1->height() + 5 * dy);
        }
        else
        {
            w1->move (w1->x() + 5 * dx, w1->y() + 5 * dy);
        }
    }
    else
    {
        if (ke->key() == Key_Enter || ke->key() == Key_Return)
            switchWidgets();
        else
        if (ke->key() == Key_M)
            switchMaskStatus();
        else
        if (ke->key() == Key_B)
        {
            if (erasePixmap())
                setBackgroundMode (PaletteBackground);
            else
                setErasePixmap (bgPixmap);
        }
        else
        if (ke->key() == Key_L)
            logging = !logging;
        else
        if (ke->key() == Key_Plus || ke->key() == Key_Equal)
        {
            if (ke->state() & QEvent::ShiftButton)
                w1->raise();
            else
            {
                QObjectList ch (*children());
                ch.find (w1);
                while (ch.next() && !ch.current()->isWidgetType());
                while (ch.next() && !ch.current()->isWidgetType());
                if (ch.current())
                    w1->stackUnder ((QWidget *) ch.current());
            }
        }
        else
        if (ke->key() == Key_Minus || ke->key() == Key_Underscore)
        {
            if (ke->state() & QEvent::ShiftButton)
                w1->lower();
            else
            {
                QObjectList ch (*children());
                ch.find (w1);
                while (ch.prev() && !ch.current()->isWidgetType());
                if (ch.current())
                    w1->stackUnder ((QWidget *) ch.current());
            }
        }
        else
        if (ke->key() == Key_V)
            w1->setShown (!w1->isShown());
    }
}

~MyWidget()
{
}

public slots:
    
private:

    MySimpleWidget *m0, *m1, *m11, *m2, *m3;
    MyPushButton *b0, *b1, *b11, *b2, *b3;
    QWidget *w1;
    
    QPixmap bgPixmap;
    bool logging;
};

bool MySimpleWidget::isLogging()
{
    if (parentWidget()->inherits ("MyWidget"))
        return ((MyWidget *) parentWidget())->isLogging();
    return false;
}

////////////////////////////////////////////////////////////////////////////////

int main (int argc, char **argv)
{
    QApplication a (argc, argv);
    MyWidget myWidget;
    
    qDebug( "Enter:             switch between simple widgets and push buttons\n"
            "Arrows:            move red widget (My1/Push1)\n"
            "Shift+Arrows:      resize red widget (My1/Push1)\n"
            "Alt+Arrows:        scroll red widget with children (My1 only)\n"
            "Shift+Alt+Arrows:  scroll red widget w/o children (My1 only)\n"
            "Ctrl+Arrows:       resize main widget\n"
            "B:                 turn background pixmap on/off\n"
            "M:                 turn widget masks on/off\n"
            "V:                 turn red widget's (My1/Push1) visibity on/off\n"
            "[+]/[-]:           move red widget up/down in z-order\n"
            "Shift+[+]/[-]:     raise/lower red widget (My1/Push1)\n"
            "L:                 turn event logging on/off\n"
            "Space:             print a separator line\n" );
    
    a.setMainWidget (&myWidget);
    myWidget.show();
    return a.exec();
}

#include "winmask.moc"

